/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.scheduledjob.scheduler.impl;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.util.concurrent.DefaultThreadFactory;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.utils.FutureThreadInfo;
import net.bluemind.scheduledjob.api.InProgressException;
import net.bluemind.scheduledjob.api.JobExitStatus;
import net.bluemind.scheduledjob.api.LogEntry;
import net.bluemind.scheduledjob.api.LogLevel;
import net.bluemind.scheduledjob.scheduler.IRecordingListener;
import net.bluemind.scheduledjob.scheduler.IScheduledJob;
import net.bluemind.scheduledjob.scheduler.IScheduledJobRunId;
import net.bluemind.scheduledjob.scheduler.IScheduler;

public class Scheduler implements IScheduler, IRecordingListener {

	private static final Logger logger = LoggerFactory.getLogger(Scheduler.class);
	private static final int MAX_SCHEDULER_THREADS = 8;

	private static final Scheduler sched = new Scheduler();
	private Map<String, RunIdImpl> activeSlots;
	private ExecutorService pool;
	private ThreadLocal<String> activeGroup;
	private ConcurrentHashMap<String, ExecutionRecorder> activeRecorders;
	private Executor logRecorderExecutor;
	private Map<String, FutureThreadInfo> runningTasks = new ConcurrentHashMap<>();
	private ReentrantLock dblock = new ReentrantLock();

	private Scheduler() {
		logRecorderExecutor = Executors.newFixedThreadPool(4, new DefaultThreadFactory("bm-scheduler-log-recorder"));

		activeGroup = new ThreadLocal<>();
		activeSlots = new ConcurrentHashMap<>();
		activeRecorders = new ConcurrentHashMap<>();
		pool = Executors.newFixedThreadPool(Math.min(MAX_SCHEDULER_THREADS, Runtime.getRuntime().availableProcessors()),
				new DefaultThreadFactory("bm-scheduler-pool"));
	}

	@Override
	public IScheduledJobRunId requestSlot(String domainName, IScheduledJob bj, Date startDate) throws ServerFault {
		String runId = domainName + "-" + bj.getJobId();
		activeSlotCheck(runId);
		RunIdImpl rid = new RunIdImpl(activeGroup.get(), domainName, bj.getJobId(), startDate);
		activeSlots.put(runId, rid);

		ExecutionRecorder recorder = new ExecutionRecorder(rid, this);
		logRecorderExecutor.execute(recorder);
		activeRecorders.put(runId, recorder);
		return rid;
	}

	private void activeSlotCheck(String runId) throws InProgressException {
		logger.debug("*** checking if {} is active", runId);
		boolean ret = activeSlots.containsKey(runId);
		if (ret) {
			logger.warn("{} was already in progress", runId);
			throw new InProgressException();
		} else if (logger.isDebugEnabled()) {
			logger.debug("      * '{}' was not in progress", runId);
		}
	}

	@Override
	public void info(IScheduledJobRunId rid, String locale, String logEntry) {
		// we log as debug here as we don't wan't un-important (& visible in the
		// ui) stuff in our system logs
		logger.debug("[{}] [{}] => {}", rid, locale, logEntry);

		log(rid, LogLevel.INFO, locale, logEntry);
	}

	@Override
	public void warn(IScheduledJobRunId rid, String locale, String logEntry) {
		logger.debug("[{}] [{}] => {}", rid, locale, logEntry);
		log(rid, LogLevel.WARNING, locale, logEntry);
	}

	private void log(IScheduledJobRunId rid, LogLevel severity, String locale, String logEntry) {
		RunIdImpl rrid = (RunIdImpl) rid;
		LogEntry le = new LogEntry();
		le.timestamp = System.currentTimeMillis();
		if (locale != null) {
			le.locale = locale;
		}
		le.severity = severity;
		le.content = logEntry != null ? logEntry : "";
		rrid.addEntry(le);
		String key = rrid.domainUid + "-" + rrid.jid;
		activeRecorders.get(key).logEntries.offer(le);
	}

	@Override
	public void error(IScheduledJobRunId rid, String locale, String logEntry) {
		logger.debug("[{}] [{}] => {}", rid, locale, logEntry);
		log(rid, LogLevel.ERROR, locale, logEntry);
	}

	@Override
	public void reportProgress(IScheduledJobRunId rid, int percent) {
		logger.debug("[{}] progress is now {}%", rid, percent);
		log(rid, LogLevel.PROGRESS, null, "#progress " + percent);
	}

	@Override
	public synchronized void finish(IScheduledJobRunId irid, JobExitStatus status) {
		RunIdImpl rid = (RunIdImpl) irid;
		logger.debug("Finishing {}", rid);
		if (rid.endTime != rid.startTime) {
			if (logger.isDebugEnabled()) {
				logger.debug("Already finished job {}", rid.jid, new Throwable());
			}
			return;
		}
		if (status == JobExitStatus.FAILURE) {
			activeSlots.remove(rid.domainUid + "-" + rid.jid);
			logger.error("finished with FAILURE status called from here", new Throwable("sched.finish(FAILURE)"));
		}
		long endStamp = System.currentTimeMillis();
		reportProgress(rid, 100);
		rid.status = status;
		rid.endTime = endStamp;

		RunIdImpl copy = rid.copy();

		String key = rid.domainUid + "-" + rid.jid;
		activeRecorders.get(key).finish();

		SendReport sr = new SendReport(copy);
		logRecorderExecutor.execute(sr);
	}

	@Override
	public void recordingComplete(RunIdImpl rid) {
		String k = rid.domainUid + "-" + rid.jid;
		long rt = rid.endTime - rid.startTime;
		logger.info("[{}] finished and recorded: {}, duration: {}ms.", rid, rid.status, rt);

		// help the live logs viewer see logs
		try {
			Thread.sleep(6000);
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		}
		activeSlots.remove(k);
		rid.destroy();
	}

	public static Scheduler get() {
		return sched;
	}

	public IScheduledJobRunId getActiveSlot(String domainName, String jid) {
		return activeSlots.get(domainName + "-" + jid);
	}

	public void tryRun(JobTicker runner) {
		String key = runner.domainName + "-" + runner.bj.getJobId();
		if (getActiveSlot(runner.domainName, runner.bj.getJobId()) == null) {
			Future<?> future = pool.submit(runner);
			runningTasks.put(key, new FutureThreadInfo(future, runner));
		}
	}

	public synchronized void cancel(String domainName, String jid) {
		String key = domainName + "-" + jid;
		logger.info("Cancelling job {}", key);
		if (runningTasks.containsKey(key)) {
			runningTasks.get(key).runnable.cancel();
			runningTasks.get(key).future.cancel(true);
		} else {
			logger.info("No running task registered for job {}", key);
		}
	}

	public void setActiveGroup(String execGroup) {
		activeGroup.set(execGroup);
	}

	/**
	 * Returns a copy of the running jobs list
	 * 
	 * @return
	 */
	public Map<String, RunIdImpl> getActiveSlots() {
		HashMap<String, RunIdImpl> copy = new HashMap<>();
		copy.putAll(activeSlots);
		return copy;
	}

	public void unregister(String domainName, IScheduledJob bj) {
		String key = domainName + "-" + bj.getJobId();
		runningTasks.remove(key);
	}

	/**
	 * Locking system for Maintenance/DataProtect
	 */
	public ReentrantLock getDbLock() {
		return dblock;
	}
}
