/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices;

import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodeRole;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.ValidationException;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.Index;

public class ShardLimitValidator {
    public static final Setting<Integer> SETTING_CLUSTER_MAX_SHARDS_PER_NODE = Setting.intSetting("cluster.max_shards_per_node", 1000, 1, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Integer> SETTING_CLUSTER_MAX_SHARDS_PER_NODE_FROZEN = Setting.intSetting("cluster.max_shards_per_node.frozen", 3000, 1, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final String FROZEN_GROUP = "frozen";
    static final Set<String> VALID_GROUPS = org.elasticsearch.core.Set.of("normal", "frozen");
    public static final Setting<String> INDEX_SETTING_SHARD_LIMIT_GROUP = Setting.simpleString("index.shard_limit.group", "normal", value -> {
        if (!VALID_GROUPS.contains(value)) {
            throw new IllegalArgumentException("[" + value + "] is not a valid shard limit group");
        }
    }, Setting.Property.IndexScope, Setting.Property.PrivateIndex, Setting.Property.NotCopyableOnResize);
    protected final AtomicInteger shardLimitPerNode = new AtomicInteger();
    protected final AtomicInteger shardLimitPerNodeFrozen = new AtomicInteger();

    public ShardLimitValidator(Settings settings, ClusterService clusterService) {
        this.shardLimitPerNode.set(SETTING_CLUSTER_MAX_SHARDS_PER_NODE.get(settings));
        this.shardLimitPerNodeFrozen.set(SETTING_CLUSTER_MAX_SHARDS_PER_NODE_FROZEN.get(settings));
        clusterService.getClusterSettings().addSettingsUpdateConsumer(SETTING_CLUSTER_MAX_SHARDS_PER_NODE, this::setShardLimitPerNode);
        clusterService.getClusterSettings().addSettingsUpdateConsumer(SETTING_CLUSTER_MAX_SHARDS_PER_NODE_FROZEN, this::setShardLimitPerNodeFrozen);
    }

    private void setShardLimitPerNode(int newValue) {
        this.shardLimitPerNode.set(newValue);
    }

    private void setShardLimitPerNodeFrozen(int newValue) {
        this.shardLimitPerNodeFrozen.set(newValue);
    }

    public int getShardLimitPerNode() {
        return this.shardLimitPerNode.get();
    }

    public void validateShardLimit(Settings settings, ClusterState state) {
        int numberOfShards = IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(settings);
        int numberOfReplicas = IndexMetadata.INDEX_NUMBER_OF_REPLICAS_SETTING.get(settings);
        int shardsToCreate = numberOfShards * (1 + numberOfReplicas);
        boolean frozen = FROZEN_GROUP.equals(INDEX_SETTING_SHARD_LIMIT_GROUP.get(settings));
        Optional<String> shardLimit = this.checkShardLimit(!frozen ? shardsToCreate : 0, frozen ? shardsToCreate : 0, state);
        if (shardLimit.isPresent()) {
            ValidationException e = new ValidationException();
            e.addValidationError(shardLimit.get());
            throw e;
        }
    }

    public void validateShardLimit(ClusterState currentState, Index[] indicesToOpen) {
        int frozen = 0;
        int normal = 0;
        for (Index index : indicesToOpen) {
            IndexMetadata imd = currentState.metadata().index(index);
            if (!imd.getState().equals((Object)IndexMetadata.State.CLOSE)) continue;
            int totalNewShards = imd.getNumberOfShards() * (1 + imd.getNumberOfReplicas());
            if (FROZEN_GROUP.equals(INDEX_SETTING_SHARD_LIMIT_GROUP.get(imd.getSettings()))) {
                frozen += totalNewShards;
                continue;
            }
            normal += totalNewShards;
        }
        Optional<String> error = this.checkShardLimit(normal, frozen, currentState);
        if (error.isPresent()) {
            ValidationException ex = new ValidationException();
            ex.addValidationError(error.get());
            throw ex;
        }
    }

    public void validateShardLimitOnReplicaUpdate(ClusterState currentState, Index[] indices, int replicas) {
        int frozen = 0;
        int normal = 0;
        for (Index index : indices) {
            IndexMetadata imd = currentState.metadata().index(index);
            int totalNewShards = this.getTotalNewShards(index, currentState, replicas);
            if (FROZEN_GROUP.equals(INDEX_SETTING_SHARD_LIMIT_GROUP.get(imd.getSettings()))) {
                frozen += totalNewShards;
                continue;
            }
            normal += totalNewShards;
        }
        Optional<String> error = this.checkShardLimit(normal, frozen, currentState);
        if (error.isPresent()) {
            ValidationException ex = new ValidationException();
            ex.addValidationError(error.get());
            throw ex;
        }
    }

    private int getTotalNewShards(Index index, ClusterState currentState, int updatedNumberOfReplicas) {
        IndexMetadata indexMetadata = currentState.metadata().index(index);
        int shardsInIndex = indexMetadata.getNumberOfShards();
        int oldNumberOfReplicas = indexMetadata.getNumberOfReplicas();
        int replicaIncrease = updatedNumberOfReplicas - oldNumberOfReplicas;
        return replicaIncrease * shardsInIndex;
    }

    private Optional<String> checkShardLimit(int newShards, int newFrozenShards, ClusterState state) {
        int frozenNodeCount = ShardLimitValidator.nodeCount(state, ShardLimitValidator::hasFrozen);
        int normalNodeCount = ShardLimitValidator.nodeCount(state, ShardLimitValidator::hasNonFrozen);
        Optional<String> normal = ShardLimitValidator.checkShardLimit(newShards, state, this.getShardLimitPerNode(), normalNodeCount, "normal");
        return normal.isPresent() ? normal : ShardLimitValidator.checkShardLimit(newFrozenShards, state, this.shardLimitPerNodeFrozen.get(), frozenNodeCount, FROZEN_GROUP);
    }

    public static boolean canAddShardsToCluster(int numberOfNewShards, int replicas, ClusterState state, boolean frozenNodes) {
        Settings clusterSettings = state.getMetadata().settings();
        int maxShardsPerNode = frozenNodes ? SETTING_CLUSTER_MAX_SHARDS_PER_NODE_FROZEN.get(clusterSettings) : SETTING_CLUSTER_MAX_SHARDS_PER_NODE.get(clusterSettings);
        int nodeCount = ShardLimitValidator.nodeCount(state, frozenNodes ? ShardLimitValidator::hasFrozen : ShardLimitValidator::hasNonFrozen);
        String nodeGroup = frozenNodes ? FROZEN_GROUP : "normal";
        Optional<String> errorMessage = ShardLimitValidator.checkShardLimit(numberOfNewShards * (1 + replicas), state, maxShardsPerNode, nodeCount, nodeGroup);
        return !errorMessage.isPresent();
    }

    static Optional<String> checkShardLimit(int newShards, ClusterState state, int maxShardsPerNode, int nodeCount, String group) {
        if (nodeCount == 0 || newShards <= 0) {
            return Optional.empty();
        }
        int maxShardsInCluster = maxShardsPerNode * nodeCount;
        int currentOpenShards = state.getMetadata().getTotalOpenIndexShards();
        if (currentOpenShards + newShards > maxShardsInCluster) {
            Predicate<IndexMetadata> indexMetadataPredicate = imd -> imd.getState().equals((Object)IndexMetadata.State.OPEN) && group.equals(INDEX_SETTING_SHARD_LIMIT_GROUP.get(imd.getSettings()));
            long currentFilteredShards = state.metadata().indices().values().stream().filter(indexMetadataPredicate).mapToInt(IndexMetadata::getTotalNumberOfShards).sum();
            if (currentFilteredShards + (long)newShards > (long)maxShardsInCluster) {
                String errorMessage = "this action would add [" + newShards + "] shards, but this cluster currently has [" + currentFilteredShards + "]/[" + maxShardsInCluster + "] maximum " + group + " shards open";
                return Optional.of(errorMessage);
            }
        }
        return Optional.empty();
    }

    private static int nodeCount(ClusterState state, Predicate<DiscoveryNode> nodePredicate) {
        return (int)state.getNodes().getDataNodes().values().stream().filter(nodePredicate).count();
    }

    private static boolean hasFrozen(DiscoveryNode node) {
        return node.getRoles().contains(DiscoveryNodeRole.DATA_FROZEN_NODE_ROLE);
    }

    private static boolean hasNonFrozen(DiscoveryNode node) {
        return node.getRoles().stream().anyMatch(r -> r.canContainData() && r != DiscoveryNodeRole.DATA_FROZEN_NODE_ROLE);
    }
}

