/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.action.bulk;

import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiConsumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.DocWriteRequest;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.bulk.Retry2;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;

public class BulkProcessor2 {
    private final int maxActionsPerBulkRequest;
    private final long maxBulkSizeBytes;
    private final ByteSizeValue maxBytesInFlight;
    private final AtomicLong totalBytesInFlight = new AtomicLong(0L);
    private volatile Scheduler.Cancellable cancellableFlushTask = null;
    private final AtomicLong executionIdGen = new AtomicLong();
    private static final Logger logger = LogManager.getLogger(BulkProcessor2.class);
    private final BiConsumer<BulkRequest, ActionListener<BulkResponse>> consumer;
    private final Listener listener;
    private final Retry2 retry;
    private final TimeValue flushInterval;
    private final ThreadPool threadPool;
    private BulkRequest bulkRequestUnderConstruction;
    private volatile boolean closed = false;
    private final Object mutex = new Object();

    public static Builder builder(BiConsumer<BulkRequest, ActionListener<BulkResponse>> consumer, Listener listener, ThreadPool threadPool) {
        Objects.requireNonNull(consumer, "consumer");
        Objects.requireNonNull(listener, "listener");
        return new Builder(consumer, listener, threadPool);
    }

    BulkProcessor2(BiConsumer<BulkRequest, ActionListener<BulkResponse>> consumer, int maxNumberOfRetries, Listener listener, int maxActionsPerBulkRequest, ByteSizeValue maxBulkSize, ByteSizeValue maxBytesInFlight, @Nullable TimeValue flushInterval, ThreadPool threadPool) {
        this.maxActionsPerBulkRequest = maxActionsPerBulkRequest;
        this.maxBulkSizeBytes = maxBulkSize.getBytes();
        this.maxBytesInFlight = maxBytesInFlight;
        this.bulkRequestUnderConstruction = new BulkRequest();
        this.consumer = consumer;
        this.listener = listener;
        this.retry = new Retry2(maxNumberOfRetries);
        this.flushInterval = flushInterval;
        this.threadPool = threadPool;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void awaitClose(long timeout, TimeUnit unit) throws InterruptedException {
        Object object = this.mutex;
        synchronized (object) {
            if (this.closed) {
                return;
            }
            this.closed = true;
            if (this.cancellableFlushTask != null) {
                this.cancellableFlushTask.cancel();
            }
            if (this.bulkRequestUnderConstruction.numberOfActions() > 0) {
                this.execute();
            }
            this.retry.awaitClose(timeout, unit);
        }
    }

    public BulkProcessor2 add(IndexRequest request) throws EsRejectedExecutionException {
        return this.add((DocWriteRequest<?>)request);
    }

    public BulkProcessor2 add(DeleteRequest request) throws EsRejectedExecutionException {
        return this.add((DocWriteRequest<?>)request);
    }

    private BulkProcessor2 add(DocWriteRequest<?> request) throws EsRejectedExecutionException {
        this.internalAdd(request);
        return this;
    }

    long getTotalBytesInFlight() {
        return this.totalBytesInFlight.get();
    }

    protected void ensureOpen() {
        if (this.closed) {
            throw new IllegalStateException("bulk process already closed");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void internalAdd(DocWriteRequest<?> request) throws EsRejectedExecutionException {
        Tuple<BulkRequest, Long> bulkRequestToExecute;
        Object object = this.mutex;
        synchronized (object) {
            this.ensureOpen();
            if (this.totalBytesInFlight.get() >= this.maxBytesInFlight.getBytes()) {
                throw new EsRejectedExecutionException("Cannot index request of size " + this.bulkRequestUnderConstruction.estimatedSizeInBytes() + " because " + this.totalBytesInFlight.get() + " bytes are already in flight and the max is " + this.maxBytesInFlight);
            }
            long bytesBeforeNewRequest = this.bulkRequestUnderConstruction.estimatedSizeInBytes();
            this.bulkRequestUnderConstruction.add(request);
            this.totalBytesInFlight.addAndGet(this.bulkRequestUnderConstruction.estimatedSizeInBytes() - bytesBeforeNewRequest);
            bulkRequestToExecute = this.newBulkRequestIfNeeded();
        }
        if (bulkRequestToExecute != null) {
            this.execute(bulkRequestToExecute.v1(), bulkRequestToExecute.v2());
        }
        this.scheduleFlushTask();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void scheduleFlushTask() {
        if (this.flushInterval == null) {
            return;
        }
        Object object = this.mutex;
        synchronized (object) {
            if (this.cancellableFlushTask == null) {
                this.cancellableFlushTask = this.threadPool.schedule(() -> {
                    Object object = this.mutex;
                    synchronized (object) {
                        if (!this.closed && this.bulkRequestUnderConstruction.numberOfActions() > 0) {
                            this.execute();
                        }
                        this.cancellableFlushTask = null;
                    }
                }, this.flushInterval, "generic");
            }
        }
    }

    private Tuple<BulkRequest, Long> newBulkRequestIfNeeded() {
        assert (Thread.holdsLock(this.mutex));
        this.ensureOpen();
        if (this.bulkRequestExceedsLimits() || this.totalBytesInFlight.get() >= this.maxBytesInFlight.getBytes()) {
            BulkRequest bulkRequest = this.bulkRequestUnderConstruction;
            this.bulkRequestUnderConstruction = new BulkRequest();
            return new Tuple<BulkRequest, Long>(bulkRequest, this.executionIdGen.incrementAndGet());
        }
        return null;
    }

    private void execute(final BulkRequest bulkRequest, final long executionId) {
        try {
            this.listener.beforeBulk(executionId, bulkRequest);
            this.retry.consumeRequestWithRetries(this.consumer, bulkRequest, new ActionListener<BulkResponse>(){

                @Override
                public void onResponse(BulkResponse response) {
                    BulkProcessor2.this.totalBytesInFlight.addAndGet(-1L * bulkRequest.estimatedSizeInBytes());
                    BulkProcessor2.this.listener.afterBulk(executionId, bulkRequest, response);
                }

                @Override
                public void onFailure(Exception e) {
                    BulkProcessor2.this.totalBytesInFlight.addAndGet(-1L * bulkRequest.estimatedSizeInBytes());
                    BulkProcessor2.this.listener.afterBulk(executionId, bulkRequest, e);
                }
            });
        }
        catch (Exception e) {
            logger.warn(() -> "Failed to execute bulk request " + executionId + ".", (Throwable)e);
            this.totalBytesInFlight.addAndGet(-1L * bulkRequest.estimatedSizeInBytes());
            this.listener.afterBulk(executionId, bulkRequest, e);
        }
    }

    private void execute() {
        assert (Thread.holdsLock(this.mutex));
        BulkRequest bulkRequest = this.bulkRequestUnderConstruction;
        long executionId = this.executionIdGen.incrementAndGet();
        this.bulkRequestUnderConstruction = new BulkRequest();
        this.execute(bulkRequest, executionId);
    }

    private boolean bulkRequestExceedsLimits() {
        assert (Thread.holdsLock(this.mutex));
        if (this.maxActionsPerBulkRequest != -1 && this.bulkRequestUnderConstruction.numberOfActions() >= this.maxActionsPerBulkRequest) {
            return true;
        }
        return this.maxBulkSizeBytes != -1L && this.bulkRequestUnderConstruction.estimatedSizeInBytes() >= this.maxBulkSizeBytes;
    }

    public static interface Listener {
        public void beforeBulk(long var1, BulkRequest var3);

        public void afterBulk(long var1, BulkRequest var3, BulkResponse var4);

        public void afterBulk(long var1, BulkRequest var3, Exception var4);
    }

    public static class Builder {
        private final BiConsumer<BulkRequest, ActionListener<BulkResponse>> consumer;
        private final Listener listener;
        private final ThreadPool threadPool;
        private int maxRequestsInBulk = 1000;
        private ByteSizeValue maxBulkSizeInBytes = new ByteSizeValue(5L, ByteSizeUnit.MB);
        private ByteSizeValue maxBytesInFlight = new ByteSizeValue(50L, ByteSizeUnit.MB);
        private TimeValue flushInterval = null;
        private int maxNumberOfRetries = 3;

        private Builder(BiConsumer<BulkRequest, ActionListener<BulkResponse>> consumer, Listener listener, ThreadPool threadPool) {
            this.consumer = consumer;
            this.listener = listener;
            this.threadPool = threadPool;
        }

        public Builder setBulkActions(int bulkActions) {
            this.maxRequestsInBulk = bulkActions;
            return this;
        }

        public Builder setBulkSize(ByteSizeValue maxBulkSizeInBytes) {
            this.maxBulkSizeInBytes = maxBulkSizeInBytes;
            return this;
        }

        public Builder setFlushInterval(TimeValue flushInterval) {
            this.flushInterval = flushInterval;
            return this;
        }

        public Builder setMaxNumberOfRetries(int maxNumberOfRetries) {
            assert (maxNumberOfRetries >= 0);
            this.maxNumberOfRetries = maxNumberOfRetries;
            return this;
        }

        public Builder setMaxBytesInFlight(ByteSizeValue maxBytesInFlight) {
            this.maxBytesInFlight = maxBytesInFlight;
            return this;
        }

        public BulkProcessor2 build() {
            return new BulkProcessor2(this.consumer, this.maxNumberOfRetries, this.listener, this.maxRequestsInBulk, this.maxBulkSizeInBytes, this.maxBytesInFlight, this.flushInterval, this.threadPool);
        }
    }
}

