/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import org.apache.kafka.clients.consumer.OffsetResetStrategy;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.streams.StreamsConfig;
import org.apache.kafka.streams.errors.TopologyException;
import org.apache.kafka.streams.processor.StateStore;
import org.apache.kafka.streams.processor.TaskId;
import org.apache.kafka.streams.processor.internals.InternalTopologyBuilder;
import org.apache.kafka.streams.processor.internals.ProcessorTopology;
import org.apache.kafka.streams.processor.internals.StreamThread;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TopologyMetadata {
    private final Logger log = LoggerFactory.getLogger(TopologyMetadata.class);
    private static final String UNNAMED_TOPOLOGY = "__UNNAMED_TOPOLOGY__";
    private static final Pattern EMPTY_ZERO_LENGTH_PATTERN = Pattern.compile("");
    private final StreamsConfig config;
    private final TopologyVersion version;
    private final ConcurrentNavigableMap<String, InternalTopologyBuilder> builders;
    private ProcessorTopology globalTopology;
    private final Map<String, StateStore> globalStateStores = new HashMap<String, StateStore>();
    private final Set<String> allInputTopics = new HashSet<String>();

    public TopologyMetadata(InternalTopologyBuilder builder, StreamsConfig config) {
        this.version = new TopologyVersion();
        this.config = config;
        this.builders = new ConcurrentSkipListMap<String, InternalTopologyBuilder>();
        if (builder.hasNamedTopology()) {
            this.builders.put(builder.topologyName(), builder);
        } else {
            this.builders.put(UNNAMED_TOPOLOGY, builder);
        }
    }

    public TopologyMetadata(ConcurrentNavigableMap<String, InternalTopologyBuilder> builders, StreamsConfig config) {
        this.version = new TopologyVersion();
        this.config = config;
        this.builders = builders;
        if (builders.isEmpty()) {
            this.log.debug("Starting up empty KafkaStreams app with no topology");
        }
    }

    public long topologyVersion() {
        return this.version.topologyVersion.get();
    }

    private void lock() {
        this.version.topologyLock.lock();
    }

    private void unlock() {
        this.version.topologyLock.unlock();
    }

    public void wakeupThreads() {
        try {
            this.lock();
            this.version.topologyCV.signalAll();
        }
        finally {
            this.unlock();
        }
    }

    public void maybeWaitForNonEmptyTopology(Supplier<StreamThread.State> threadState) {
        if (this.isEmpty() && threadState.get().isAlive()) {
            try {
                this.lock();
                while (this.isEmpty() && threadState.get().isAlive()) {
                    try {
                        this.log.debug("Detected that the topology is currently empty, waiting for something to process");
                        this.version.topologyCV.await();
                    }
                    catch (InterruptedException e) {
                        this.log.debug("StreamThread was interrupted while waiting on empty topology", (Throwable)e);
                    }
                }
            }
            finally {
                this.unlock();
            }
        }
    }

    public void registerAndBuildNewTopology(InternalTopologyBuilder newTopologyBuilder) {
        try {
            this.lock();
            this.version.topologyVersion.incrementAndGet();
            this.log.info("Adding NamedTopology {}, latest topology version is {}", (Object)newTopologyBuilder.topologyName(), (Object)this.version.topologyVersion.get());
            this.builders.put(newTopologyBuilder.topologyName(), newTopologyBuilder);
            this.buildAndVerifyTopology(newTopologyBuilder);
            this.version.topologyCV.signalAll();
        }
        finally {
            this.unlock();
        }
    }

    public void unregisterTopology(String topologyName) {
        try {
            this.lock();
            this.version.topologyVersion.incrementAndGet();
            this.log.info("Removing NamedTopology {}, latest topology version is {}", (Object)topologyName, (Object)this.version.topologyVersion.get());
            InternalTopologyBuilder removedBuilder = (InternalTopologyBuilder)this.builders.remove(topologyName);
            removedBuilder.fullSourceTopicNames().forEach(this.allInputTopics::remove);
            removedBuilder.allSourcePatternStrings().forEach(this.allInputTopics::remove);
            this.version.topologyCV.signalAll();
        }
        finally {
            this.unlock();
        }
    }

    public void buildAndRewriteTopology() {
        this.applyToEachBuilder(this::buildAndVerifyTopology);
    }

    private void buildAndVerifyTopology(InternalTopologyBuilder builder) {
        builder.rewriteTopology(this.config);
        builder.buildTopology();
        int numInputTopics = this.allInputTopics.size();
        List<String> inputTopics = builder.fullSourceTopicNames();
        List<String> inputPatterns = builder.allSourcePatternStrings();
        int numNewInputTopics = inputTopics.size() + inputPatterns.size();
        this.allInputTopics.addAll(inputTopics);
        this.allInputTopics.addAll(inputPatterns);
        if (this.allInputTopics.size() != numInputTopics + numNewInputTopics) {
            inputTopics.retainAll(this.allInputTopics);
            inputPatterns.retainAll(this.allInputTopics);
            this.log.error("Tried to add the NamedTopology {} but it had overlap with other input topics {} or patterns {}", new Object[]{builder.topologyName(), inputTopics, inputPatterns});
            throw new TopologyException("Named Topologies may not subscribe to the same input topics or patterns");
        }
        ProcessorTopology globalTopology = builder.buildGlobalStateTopology();
        if (globalTopology != null) {
            if (builder.topologyName() != null) {
                throw new IllegalStateException("Global state stores are not supported with Named Topologies");
            }
            if (this.globalTopology == null) {
                this.globalTopology = globalTopology;
            } else {
                throw new IllegalStateException("Topology builder had global state, but global topology has already been set");
            }
        }
        this.globalStateStores.putAll(builder.globalStateStores());
    }

    public int getNumStreamThreads(StreamsConfig config) {
        int configuredNumStreamThreads = config.getInt("num.stream.threads");
        if (this.hasNamedTopologies()) {
            if (this.hasNoLocalTopology()) {
                this.log.error("Detected a named topology with no input topics, a named topology may not be empty.");
                throw new TopologyException("Topology has no stream threads and no global threads, must subscribe to at least one source topic or pattern.");
            }
        } else if (this.hasNoLocalTopology() && !this.hasGlobalTopology()) {
            this.log.error("Topology with no input topics will create no stream threads and no global thread.");
            throw new TopologyException("Topology has no stream threads and no global threads, must subscribe to at least one source topic or global table.");
        }
        if (configuredNumStreamThreads != 0 && this.hasNoLocalTopology()) {
            this.log.info("Overriding number of StreamThreads to zero for global-only topology");
            return 0;
        }
        return configuredNumStreamThreads;
    }

    public boolean hasNamedTopologies() {
        return !this.builders.containsKey(UNNAMED_TOPOLOGY);
    }

    Set<String> namedTopologiesView() {
        return this.hasNamedTopologies() ? Collections.unmodifiableSet(this.builders.keySet()) : Collections.emptySet();
    }

    public boolean hasGlobalTopology() {
        return this.evaluateConditionIsTrueForAnyBuilders(InternalTopologyBuilder::hasGlobalStores);
    }

    public boolean hasNoLocalTopology() {
        return this.evaluateConditionIsTrueForAnyBuilders(InternalTopologyBuilder::hasNoLocalTopology);
    }

    public boolean hasPersistentStores() {
        if (this.hasNamedTopologies()) {
            return true;
        }
        return this.evaluateConditionIsTrueForAnyBuilders(InternalTopologyBuilder::hasPersistentStores);
    }

    public boolean hasStore(String name) {
        return this.evaluateConditionIsTrueForAnyBuilders(b -> b.hasStore(name));
    }

    public boolean hasOffsetResetOverrides() {
        return this.hasNamedTopologies() || this.evaluateConditionIsTrueForAnyBuilders(InternalTopologyBuilder::hasOffsetResetOverrides);
    }

    public OffsetResetStrategy offsetResetStrategy(String topic) {
        for (InternalTopologyBuilder builder : this.builders.values()) {
            OffsetResetStrategy resetStrategy = builder.offsetResetStrategy(topic);
            if (resetStrategy == null) continue;
            return resetStrategy;
        }
        return null;
    }

    Collection<String> sourceTopicCollection() {
        ArrayList<String> sourceTopics = new ArrayList<String>();
        this.applyToEachBuilder(b -> sourceTopics.addAll(b.sourceTopicCollection()));
        return sourceTopics;
    }

    Pattern sourceTopicPattern() {
        StringBuilder patternBuilder = new StringBuilder();
        this.applyToEachBuilder(b -> {
            String patternString = b.sourceTopicsPatternString();
            if (patternString.length() > 0) {
                patternBuilder.append(patternString).append("|");
            }
        });
        if (patternBuilder.length() > 0) {
            patternBuilder.setLength(patternBuilder.length() - 1);
            return Pattern.compile(patternBuilder.toString());
        }
        return EMPTY_ZERO_LENGTH_PATTERN;
    }

    public boolean usesPatternSubscription() {
        return this.evaluateConditionIsTrueForAnyBuilders(InternalTopologyBuilder::usesPatternSubscription);
    }

    public boolean isEmpty() {
        return this.builders.isEmpty();
    }

    public String topologyDescriptionString() {
        if (this.isEmpty()) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        this.applyToEachBuilder(b -> sb.append(b.describe().toString()));
        return sb.toString();
    }

    public ProcessorTopology buildSubtopology(TaskId task) {
        InternalTopologyBuilder builder = this.lookupBuilderForTask(task);
        return builder == null ? null : builder.buildSubtopology(task.subtopology());
    }

    public ProcessorTopology globalTaskTopology() {
        if (this.hasNamedTopologies()) {
            throw new IllegalStateException("Global state stores are not supported with Named Topologies");
        }
        return this.globalTopology;
    }

    public Map<String, StateStore> globalStateStores() {
        return this.globalStateStores;
    }

    public Map<String, List<String>> stateStoreNameToSourceTopics() {
        HashMap<String, List<String>> stateStoreNameToSourceTopics = new HashMap<String, List<String>>();
        this.applyToEachBuilder(b -> stateStoreNameToSourceTopics.putAll(b.stateStoreNameToSourceTopics()));
        return stateStoreNameToSourceTopics;
    }

    public String getStoreForChangelogTopic(String topicName) {
        for (InternalTopologyBuilder builder : this.builders.values()) {
            String store = builder.getStoreForChangelogTopic(topicName);
            if (store == null) continue;
            return store;
        }
        this.log.warn("Unable to locate any store for topic {}", (Object)topicName);
        return "";
    }

    public Collection<String> sourceTopicsForStore(String storeName) {
        ArrayList<String> sourceTopics = new ArrayList<String>();
        this.applyToEachBuilder(b -> sourceTopics.addAll(b.sourceTopicsForStore(storeName)));
        return sourceTopics;
    }

    public Map<Subtopology, InternalTopologyBuilder.TopicsInfo> topicGroups() {
        HashMap<Subtopology, InternalTopologyBuilder.TopicsInfo> topicGroups = new HashMap<Subtopology, InternalTopologyBuilder.TopicsInfo>();
        this.applyToEachBuilder(b -> topicGroups.putAll(b.topicGroups()));
        return topicGroups;
    }

    public Map<String, List<String>> nodeToSourceTopics(TaskId task) {
        return this.lookupBuilderForTask(task).nodeToSourceTopics();
    }

    void addSubscribedTopicsFromMetadata(Set<String> topics, String logPrefix) {
        this.applyToEachBuilder(b -> b.addSubscribedTopicsFromMetadata(topics, logPrefix));
    }

    void addSubscribedTopicsFromAssignment(List<TopicPartition> partitions, String logPrefix) {
        this.applyToEachBuilder(b -> b.addSubscribedTopicsFromAssignment(partitions, logPrefix));
    }

    public Collection<Set<String>> copartitionGroups() {
        ArrayList<Set<String>> copartitionGroups = new ArrayList<Set<String>>();
        this.applyToEachBuilder(b -> copartitionGroups.addAll(b.copartitionGroups()));
        return copartitionGroups;
    }

    private InternalTopologyBuilder lookupBuilderForTask(TaskId task) {
        return task.topologyName() == null ? (InternalTopologyBuilder)this.builders.get(UNNAMED_TOPOLOGY) : (InternalTopologyBuilder)this.builders.get(task.topologyName());
    }

    public InternalTopologyBuilder lookupBuilderForNamedTopology(String name) {
        return (InternalTopologyBuilder)this.builders.get(name);
    }

    private boolean evaluateConditionIsTrueForAnyBuilders(Function<InternalTopologyBuilder, Boolean> condition) {
        for (InternalTopologyBuilder builder : this.builders.values()) {
            if (!condition.apply(builder).booleanValue()) continue;
            return true;
        }
        return false;
    }

    private void applyToEachBuilder(Consumer<InternalTopologyBuilder> function) {
        for (InternalTopologyBuilder builder : this.builders.values()) {
            function.accept(builder);
        }
    }

    public static class Subtopology {
        final int nodeGroupId;
        final String namedTopology;

        public Subtopology(int nodeGroupId, String namedTopology) {
            this.nodeGroupId = nodeGroupId;
            this.namedTopology = namedTopology;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Subtopology that = (Subtopology)o;
            return this.nodeGroupId == that.nodeGroupId && Objects.equals(this.namedTopology, that.namedTopology);
        }

        public int hashCode() {
            return Objects.hash(this.nodeGroupId, this.namedTopology);
        }
    }

    public static class TopologyVersion {
        public AtomicLong topologyVersion = new AtomicLong(0L);
        public ReentrantLock topologyLock = new ReentrantLock();
        public Condition topologyCV = this.topologyLock.newCondition();
    }
}

