/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.AuthenticationException;
import org.apache.kafka.common.errors.AuthorizationException;
import org.apache.kafka.common.errors.InvalidProducerEpochException;
import org.apache.kafka.common.errors.InvalidTopicException;
import org.apache.kafka.common.errors.OffsetMetadataTooLarge;
import org.apache.kafka.common.errors.OutOfOrderSequenceException;
import org.apache.kafka.common.errors.ProducerFencedException;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.errors.SecurityDisabledException;
import org.apache.kafka.common.errors.SerializationException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.errors.UnknownServerException;
import org.apache.kafka.common.header.Headers;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.serialization.Serializer;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.streams.errors.ProductionExceptionHandler;
import org.apache.kafka.streams.errors.StreamsException;
import org.apache.kafka.streams.errors.TaskCorruptedException;
import org.apache.kafka.streams.errors.TaskMigratedException;
import org.apache.kafka.streams.processor.StreamPartitioner;
import org.apache.kafka.streams.processor.TaskId;
import org.apache.kafka.streams.processor.internals.RecordCollector;
import org.apache.kafka.streams.processor.internals.StreamsProducer;
import org.apache.kafka.streams.processor.internals.metrics.StreamsMetricsImpl;
import org.apache.kafka.streams.processor.internals.metrics.TaskMetrics;
import org.slf4j.Logger;

public class RecordCollectorImpl
implements RecordCollector {
    private static final String SEND_EXCEPTION_MESSAGE = "Error encountered sending record to topic %s for task %s due to:%n%s";
    private final Logger log;
    private final TaskId taskId;
    private final StreamsProducer streamsProducer;
    private final ProductionExceptionHandler productionExceptionHandler;
    private final Sensor droppedRecordsSensor;
    private final boolean eosEnabled;
    private final Map<TopicPartition, Long> offsets;
    private final AtomicReference<KafkaException> sendException = new AtomicReference<Object>(null);

    public RecordCollectorImpl(LogContext logContext, TaskId taskId, StreamsProducer streamsProducer, ProductionExceptionHandler productionExceptionHandler, StreamsMetricsImpl streamsMetrics) {
        this.log = logContext.logger(this.getClass());
        this.taskId = taskId;
        this.streamsProducer = streamsProducer;
        this.productionExceptionHandler = productionExceptionHandler;
        this.eosEnabled = streamsProducer.eosEnabled();
        String threadId = Thread.currentThread().getName();
        this.droppedRecordsSensor = TaskMetrics.droppedRecordsSensor(threadId, taskId.toString(), streamsMetrics);
        this.offsets = new HashMap<TopicPartition, Long>();
    }

    @Override
    public void initialize() {
        if (this.eosEnabled) {
            this.streamsProducer.initTransaction();
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public <K, V> void send(String topic, K key, V value, Headers headers, Long timestamp, Serializer<K> keySerializer, Serializer<V> valueSerializer, StreamPartitioner<? super K, ? super V> partitioner) {
        Integer partition;
        if (partitioner != null) {
            List<PartitionInfo> partitions;
            try {
                partitions = this.streamsProducer.partitionsFor(topic);
            }
            catch (TimeoutException timeoutException) {
                this.log.warn("Could not get partitions for topic {}, will retry", (Object)topic);
                throw timeoutException;
            }
            catch (KafkaException fatal) {
                throw new StreamsException("Could not determine the number of partitions for topic '" + topic + "' for task " + this.taskId + " due to " + fatal.toString(), fatal);
            }
            if (partitions.size() <= 0) throw new StreamsException("Could not get partition information for topic " + topic + " for task " + this.taskId + ". This can happen if the topic does not exist.");
            partition = partitioner.partition(topic, key, value, partitions.size());
        } else {
            partition = null;
        }
        this.send(topic, key, value, headers, partition, timestamp, keySerializer, valueSerializer);
    }

    @Override
    public <K, V> void send(String topic, K key, V value, Headers headers, Integer partition, Long timestamp, Serializer<K> keySerializer, Serializer<V> valueSerializer) {
        byte[] valBytes;
        byte[] keyBytes;
        this.checkForException();
        try {
            keyBytes = keySerializer.serialize(topic, headers, key);
            valBytes = valueSerializer.serialize(topic, headers, value);
        }
        catch (ClassCastException exception2) {
            String keyClass = key == null ? "unknown because key is null" : key.getClass().getName();
            String valueClass = value == null ? "unknown because value is null" : value.getClass().getName();
            throw new StreamsException(String.format("ClassCastException while producing data to topic %s. A serializer (key: %s / value: %s) is not compatible to the actual key or value type (key type: %s / value type: %s). Change the default Serdes in StreamConfig or provide correct Serdes via method parameters (for example if using the DSL, `#to(String topic, Produced<K, V> produced)` with `Produced.keySerde(WindowedSerdes.timeWindowedSerdeFrom(String.class))`).", topic, keySerializer.getClass().getName(), valueSerializer.getClass().getName(), keyClass, valueClass), exception2);
        }
        catch (RuntimeException exception3) {
            String errorMessage = String.format(SEND_EXCEPTION_MESSAGE, topic, this.taskId, exception3.toString());
            throw new StreamsException(errorMessage, exception3);
        }
        ProducerRecord<byte[], byte[]> serializedRecord = new ProducerRecord<byte[], byte[]>(topic, partition, timestamp, keyBytes, valBytes, headers);
        this.streamsProducer.send(serializedRecord, (metadata, exception) -> {
            if (this.sendException.get() != null) {
                return;
            }
            if (exception == null) {
                TopicPartition tp = new TopicPartition(metadata.topic(), metadata.partition());
                if (metadata.offset() >= 0L) {
                    this.offsets.put(tp, metadata.offset());
                } else {
                    this.log.warn("Received offset={} in produce response for {}", (Object)metadata.offset(), (Object)tp);
                }
            } else {
                this.recordSendError(topic, exception, serializedRecord);
                this.log.trace("Failed record: (key {} value {} timestamp {}) topic=[{}] partition=[{}]", new Object[]{key, value, timestamp, topic, partition});
            }
        });
    }

    private void recordSendError(String topic, Exception exception, ProducerRecord<byte[], byte[]> serializedRecord) {
        String errorMessage = String.format(SEND_EXCEPTION_MESSAGE, topic, this.taskId, exception.toString());
        if (this.isFatalException(exception)) {
            errorMessage = errorMessage + "\nWritten offsets would not be recorded and no more records would be sent since this is a fatal error.";
            this.sendException.set(new StreamsException(errorMessage, exception));
        } else if (exception instanceof ProducerFencedException || exception instanceof InvalidProducerEpochException || exception instanceof OutOfOrderSequenceException) {
            errorMessage = errorMessage + "\nWritten offsets would not be recorded and no more records would be sent since the producer is fenced, indicating the task may be migrated out";
            this.sendException.set(new TaskMigratedException(errorMessage, exception));
        } else if (exception instanceof RetriableException) {
            errorMessage = errorMessage + "\nThe broker is either slow or in bad state (like not having enough replicas) in responding the request, or the connection to broker was interrupted sending the request or receiving the response. \nConsider overwriting `max.block.ms` and /or `delivery.timeout.ms` to a larger value to wait longer for such scenarios and avoid timeout errors";
            this.sendException.set(new TaskCorruptedException(Collections.singleton(this.taskId)));
        } else if (this.productionExceptionHandler.handle(serializedRecord, exception) == ProductionExceptionHandler.ProductionExceptionHandlerResponse.FAIL) {
            errorMessage = errorMessage + "\nException handler choose to FAIL the processing, no more records would be sent.";
            this.sendException.set(new StreamsException(errorMessage, exception));
        } else {
            errorMessage = errorMessage + "\nException handler choose to CONTINUE processing in spite of this error but written offsets would not be recorded.";
            this.droppedRecordsSensor.record();
        }
        this.log.error(errorMessage, (Throwable)exception);
    }

    private boolean isFatalException(Exception exception) {
        boolean securityException = exception instanceof AuthenticationException || exception instanceof AuthorizationException || exception instanceof SecurityDisabledException;
        boolean communicationException = exception instanceof InvalidTopicException || exception instanceof UnknownServerException || exception instanceof SerializationException || exception instanceof OffsetMetadataTooLarge || exception instanceof IllegalStateException;
        return securityException || communicationException;
    }

    @Override
    public void flush() {
        this.log.debug("Flushing record collector");
        this.streamsProducer.flush();
        this.checkForException();
    }

    @Override
    public void closeClean() {
        this.log.info("Closing record collector clean");
        this.checkForException();
    }

    @Override
    public void closeDirty() {
        this.log.info("Closing record collector dirty");
        if (this.eosEnabled) {
            this.streamsProducer.abortTransaction();
        }
        this.checkForException();
    }

    @Override
    public Map<TopicPartition, Long> offsets() {
        return Collections.unmodifiableMap(new HashMap<TopicPartition, Long>(this.offsets));
    }

    private void checkForException() {
        KafkaException exception = this.sendException.get();
        if (exception != null) {
            this.sendException.set(null);
            throw exception;
        }
    }

    Producer<byte[], byte[]> producer() {
        return this.streamsProducer.kafkaProducer();
    }
}

