/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2024
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.core.container.service.internal;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IInternalContainersByLocation;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.persistence.ContainerStore;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.jdbc.JdbcAbstractStore;
import net.bluemind.core.rest.BmContext;
import net.bluemind.i18n.labels.I18nLabels;
import net.bluemind.role.api.BasicRoles;

public class ContainersByLocation implements IInternalContainersByLocation {

	private final SecurityContext securityContext;
	private final BmContext context;
	private final DataSource dataSource;

	private static final Logger logger = LoggerFactory.getLogger(ContainersByLocation.class);

	public ContainersByLocation(BmContext context, DataSource dataSource) {
		this.securityContext = context.getSecurityContext();
		this.context = context;
		this.dataSource = dataSource;
	}

	@Override
	public List<BaseContainerDescriptor> listByType(ContainerQuery query) {
		RBACManager.forContext(context).check(BasicRoles.ROLE_ADMIN);
		checkContainerQuery(query);
		return listContainersByType(query);
	}

	private List<BaseContainerDescriptor> listContainersByType(ContainerQuery query) {
		try {
			ContainerStore containerStore = new ContainerStore(context, dataSource, securityContext);
			List<Container> containers = JdbcAbstractStore.doOrFail(() -> containerStore.findByType(query.type));
			return asDescriptorsLight(containers);
		} catch (Exception e) {
			logger.warn("Fail to fetch containers on datasource {}", dataSource);
		}
		return Collections.emptyList();
	}

	private static void checkContainerQuery(ContainerQuery query) {
		if (query.type == null || query.type.isEmpty()) {
			throw new IllegalArgumentException("Invalid request, query.type is missing");
		}
	}

	private List<BaseContainerDescriptor> asDescriptorsLight(List<Container> containers) throws ServerFault {
		List<BaseContainerDescriptor> ret = new ArrayList<>(containers.size());
		for (Container c : containers) {
			String label = I18nLabels.getInstance().translate(securityContext.getLang(), c.name);
			ret.add(BaseContainerDescriptor.create(c.uid, label, c.owner, c.type, c.domainUid, c.defaultContainer,
					c.id));
		}
		return ret;
	}

	@Override
	public List<BaseContainerDescriptor> listByTypeAndOwner(ContainerQuery query) {
		RBACManager.forContext(context).check(BasicRoles.ROLE_ADMIN);
		checkContainerQueryByOwnerAndType(query);
		return listContainersByOwnerAndType(query);
	}

	private List<BaseContainerDescriptor> listContainersByOwnerAndType(ContainerQuery query) {
		try {
			ContainerStore containerStore = new ContainerStore(context, dataSource, securityContext);
			List<Container> containers = JdbcAbstractStore
					.doOrFail(() -> containerStore.findByTypeAndOwner(query.type, query.owner));
			return asDescriptorsLight(containers);
		} catch (Exception e) {
			logger.warn("Fail to fetch containers on datasource {}", dataSource);
		}
		return Collections.emptyList();
	}

	private static void checkContainerQueryByOwnerAndType(ContainerQuery query) {
		if (query.type == null || query.type.isEmpty() || query.owner == null || query.owner.isEmpty()) {
			throw new IllegalArgumentException("Invalid request, query.type and query.owner are mandatory");
		}
	}

}
