<?php

/* BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2017
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
namespace BM;
require_once 'GlobalEnv.php';

/**
 * Implementation of net.bluemind.core.container.api.IContainerManagement.
 */

class ContainerManagementClient {

  protected $base;
  protected $sid;
  protected $containerUid;

  /*
   * Constructor.
   *
   * @param base
   * @param sid
   * @param containerUid
   *
   */
  public function __construct($base, $sid , $containerUid) {
    $this->sid = $sid;
    $this->base = $base."/api/containers/_manage/{containerUid}";
    $this->containerUid = $containerUid;
    $this->base = str_replace("{containerUid}", urlencode($containerUid), $this->base);
  }


  /*
   * @return
   */
  public function getDescriptor() {
    $postUri = "/_descriptor";
    $method = "GET";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @return
   */
  public function getSettings() {
    $postUri = "/_settings";
    $method = "GET";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @return
   */
  public function getAccessControlList() {
    $postUri = "/_acl";
    $method = "GET";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @return
   */
  public function getAllItems() {
    $postUri = "/_list";
    $method = "GET";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @return
   */
  public function subscribers() {
    $postUri = "/_subscription";
    $method = "GET";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @return
   */
  public function getItemCount() {
    $postUri = "/_itemCount";
    $method = "GET";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param entries
   * @return
   */
  public function setAccessControlList( $entries  ) {
    $postUri = "/_acl";
    $method = "PUT";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $entries;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param descriptor
   * @return
   */
  public function update( $descriptor  ) {
    $postUri = "/_descriptor";
    $method = "POST";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $descriptor;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param subject
   * @return
   */
  public function allowOfflineSync( $subject  ) {
    $postUri = "/{subject}/offlineSync";
    $method = "PUT";

    $postUri = str_replace("{subject}", urlencode($subject), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param settings
   * @return
   */
  public function setSettings( $settings  ) {
    $postUri = "/_settings";
    $method = "PUT";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $settings;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param uids
   * @return
   */
  public function getItems( $uids  ) {
    $postUri = "/_mget";
    $method = "POST";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $uids;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param subject
   * @return
   */
  public function disallowOfflineSync( $subject  ) {
    $postUri = "/{subject}/offlineSync";
    $method = "DELETE";

    $postUri = str_replace("{subject}", urlencode($subject), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param verbsOrRoles
   * @return
   */
  public function canAccess( $verbsOrRoles  ) {
    $postUri = "/_canAccess";
    $method = "POST";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $verbsOrRoles;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param settings
   * @return
   */
  public function setPersonalSettings( $settings  ) {
    $postUri = "/_personalSettings";
    $method = "PUT";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $settings;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param filter
   * @return
   */
  public function getFilteredItems( $filter  ) {
    $postUri = "/_filtered";
    $method = "GET";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $filter;
    return $this->execute($url, $queryParam, $body, $method);
  }


  /*
   * Execute the request
   *
   * @param url
   * @param data
   * @param body
   */
  private function execute($url, $queryParam, $body, $method) {

    $curl = curl_init();

    $headers = array();
    array_push($headers, 'X-BM-ApiKey: '.$this->sid);

    if (sizeof($queryParam) > 0) {
      $url .= '?'.http_build_query($queryParam);
    }

    curl_setopt_array($curl, GlobalEnv::$curlOptions + array(
      CURLOPT_URL => $url,
      CURLOPT_HTTPHEADER => $headers,
      CURLOPT_CUSTOMREQUEST => $method)
    );

    if ($method == 'POST') {
      curl_setopt($curl, CURLOPT_POST, TRUE);
      curl_setopt($curl, CURLOPT_POSTFIELDS, array());
    }

    if (is_resource($body)) {
      if ($method == 'PUT') {
        curl_setopt($curl, CURLOPT_PUT, TRUE);
      }
      $size = fstat($body)['size'];
      curl_setopt($curl, CURLOPT_INFILE, $body);
      curl_setopt($curl, CURLOPT_INFILESIZE, $size);
    } else {
      if (is_object($body) && method_exists($body, 'serialize')) {
        $body = $body->serialize();
      } else if (is_object($body)) {
        $body = json_encode($body);
      } else if (is_array($body)) {
        $body = json_encode($body);
      } else if (is_string($body)) {
        $body = json_encode($body);
      }
      $size = strlen($body);
      array_push($headers, 'Content-Type: application/json');
      array_push($headers, 'Content-Length: '.$size);
      curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);
      curl_setopt($curl, CURLOPT_POSTFIELDS, $body);
    }

    $resp = curl_exec($curl);
    if ($resp === false) {
      throw new \Exception(curl_error($curl));
    } 
    curl_close($curl);
    if (!$resp) {
      return;
    }
    $js = json_decode($resp);
    if ($js === NULL) {
      return $resp;
    }
    if (isset($js->errorCode)) {
      throw new \Exception($js->errorCode . ': ' . (isset($js->message) ? ' : ' . $js->message : ''));
    }
    return $js;
  }

}
