/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2018
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.service.internal.repair;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import com.google.common.collect.ImmutableSet;

import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailboxReplica;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.MaintenanceOperation;
import net.bluemind.directory.service.IDirEntryRepairSupport;
import net.bluemind.directory.service.RepairTaskMonitor;
import net.bluemind.mailbox.api.Mailbox;

public class MissingRecordsContainersRepair implements IDirEntryRepairSupport {

	public static class RepairFactory implements IDirEntryRepairSupport.Factory {
		@Override
		public IDirEntryRepairSupport create(BmContext context) {
			return new MissingRecordsContainersRepair(context);
		}
	}

	public static final MaintenanceOperation missingConts = MaintenanceOperation
			.create(IMailReplicaUids.REPAIR_MISSING_REC_CONTAINERS, "Record containers missing for existing replica");

	private static class MissingRecordsContainersMaintenance extends MailboxFoldersRepairOp {

		public MissingRecordsContainersMaintenance(BmContext ctx) {
			super(ctx, missingConts.identifier, null, IMailReplicaUids.REPAIR_SUBTREE_OP, 1);
		}

		@Override
		public void runOnFolders(boolean repair, RepairTaskMonitor monitor, String subTree, String domainUid,
				ItemValue<Mailbox> mbox, List<ItemValue<MailboxReplica>> fullList) {
			monitor.begin(fullList.size(), "Processing " + fullList.size() + " folders.");
			IContainers conts = context.provider().instance(IContainers.class);
			int fixed = 0;
			for (ItemValue<MailboxReplica> folder : fullList) {
				ContainerDescriptor cont = conts.getIfPresent(IMailReplicaUids.mboxRecords(folder.uid));
				if (cont == null) {
					ContainerDescriptor cd = ContainerDescriptor.create(IMailReplicaUids.mboxRecords(folder.uid),
							folder.value.fullName, mbox.uid, IMailReplicaUids.MAILBOX_RECORDS, domainUid, false);
					if (repair) {
						conts.create(cd.uid, cd);
					}
					fixed++;
				}
			}
			monitor.end(true, fixed + " container(s) were missing.", "{}");
		}

	}

	private final BmContext context;

	public MissingRecordsContainersRepair(BmContext context) {
		this.context = context;
	}

	@Override
	public Set<MaintenanceOperation> availableOperations(Kind kind) {
		if (kind == Kind.USER || kind == Kind.MAILSHARE || kind == Kind.GROUP || kind == Kind.RESOURCE) {
			return ImmutableSet.of(missingConts);
		} else {
			return Collections.emptySet();
		}
	}

	@Override
	public Set<InternalMaintenanceOperation> ops(Kind kind) {
		if (kind == Kind.USER || kind == Kind.MAILSHARE || kind == Kind.GROUP || kind == Kind.RESOURCE) {
			return ImmutableSet.of(new MissingRecordsContainersMaintenance(context));
		} else {
			return Collections.emptySet();
		}

	}
}
