/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.backend.mail.replica.service.internal.repair;

import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.ImmutableSet;

import net.bluemind.authentication.api.IAuthentication;
import net.bluemind.authentication.api.LoginResponse;
import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.replica.api.IDbMessageBodies;
import net.bluemind.backend.mail.replica.api.IDbReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.config.Token;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.persistence.ContainerStore;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.vertx.VertxStream;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.MaintenanceOperation;
import net.bluemind.directory.service.IDirEntryRepairSupport;
import net.bluemind.directory.service.RepairTaskMonitor;
import net.bluemind.imap.IMAPByteSource;
import net.bluemind.imap.IMAPException;
import net.bluemind.imap.StoreClient;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.network.topology.Topology;
import net.bluemind.server.api.Server;

public class BodiesMissingRepair implements IDirEntryRepairSupport {
	public static final MaintenanceOperation MAINTENANCE_OPERATION = MaintenanceOperation
			.create(IMailReplicaUids.REPAIR_MISSING_BODIES, "Check replication for missing t_message_bodies");
	private final BmContext context;
	private static final Logger logger = LoggerFactory.getLogger(BodiesMissingRepair.class);

	public BodiesMissingRepair(BmContext context) {
		this.context = context;
	}

	public static class RepairFactory implements IDirEntryRepairSupport.Factory {
		@Override
		public IDirEntryRepairSupport create(BmContext context) {
			return new BodiesMissingRepair(context);
		}
	}

	public static class BodiesMissingMaintenance extends InternalMaintenanceOperation {
		private final BmContext context;
		private static final String FIND_MISSING_BODIES_QUERY = "SELECT t_mailbox_record.imap_uid, "
				+ " encode(t_mailbox_record.message_body_guid, 'hex') AS message_body_guid" //
				+ " FROM t_mailbox_record LEFT JOIN t_message_body ON (t_message_body.guid = t_mailbox_record.message_body_guid)" //
				+ " WHERE container_id = ? " //
				+ " AND t_message_body.guid IS NULL AND system_flags & 4 != 4";
		// systemflags & 4 => DELETED

		public BodiesMissingMaintenance(BmContext context) {
			super(MAINTENANCE_OPERATION.identifier, null, null, 1);
			this.context = context;
		}

		public static class ItemRecords {
			public final String domainUid;
			public final ItemValue<Mailbox> mailbox;
			public final Container recordsContainer;
			public final List<ItemRecord> records;
			public String fullFolderName;
			private final IDbMessageBodies messageBodiesService;

			public ItemRecords(String domainUid, ItemValue<Mailbox> mailbox, String fullFolderName,
					Container recordsContainer, IDbMessageBodies messageBodiesService) {
				this.domainUid = domainUid;
				this.mailbox = mailbox;
				this.fullFolderName = fullFolderName;
				this.recordsContainer = recordsContainer;
				this.messageBodiesService = messageBodiesService;
				records = new ArrayList<>();
			}

			public ItemValue<Server> getServer() {
				return Topology.get().datalocation(mailbox.value.dataLocation);
			}

			public boolean shared() {
				return mailbox.value.type.sharedNs;
			}

			public ItemRecords add(int imapUid, String bodyGuid) {
				this.records.add(new ItemRecord(imapUid, bodyGuid));
				return this;
			}

			public int size() {
				return records.size();
			}

			public void uploadBody(String guid, InputStream in) {
				messageBodiesService.create(guid, VertxStream.stream(in));
			}
		}

		public static class ItemRecord {
			public final int imapUid;
			public final String bodyGuid;

			public ItemRecord(int imapUid, String bodyGuid) {
				this.imapUid = imapUid;
				this.bodyGuid = bodyGuid;
			}
		}

		@Override
		public void check(String domainUid, DirEntry entry, RepairTaskMonitor monitor) {
			String logId = entry.email != null ? entry.email : entry.displayName;
			monitor.log("[{}] check mailbox bodies missing", logId);
			monitor.end(this.execute(domainUid, entry, logId, toRepair -> {
			}), null, null);
		}

		@Override
		public void repair(String domainUid, DirEntry entry, RepairTaskMonitor monitor) {
			String logId = entry.email != null ? entry.email : entry.displayName;
			logger.info("[{}] repair mailbox bodies missing", logId);
			monitor.end(this.execute(domainUid, entry, logId, toRepair -> {
				if (toRepair.size() > 0) {
					repairRecordContainerItems(logId, toRepair);
				} else {
					logger.info("[{}] no bodies missing", logId);
				}
			}), null, null);
		}

		private boolean execute(String domainUid, DirEntry entry, String logId, Consumer<ItemRecords> runOp) {
			IMailboxes mboxApi = context.provider().instance(IMailboxes.class, domainUid);
			ItemValue<Mailbox> mailbox = mboxApi.getComplete(entry.entryUid);
			if (mailbox == null) {
				logger.error("[{}]: no mailbox, nothing to repair", logId);
				return false;
			}
			if (mailbox.value.dataLocation == null) {
				logger.error("[{}] no dataLocation, can't repair", mailbox);
				return false;
			}
			DataSource ds = context.getMailboxDataSource(entry.dataLocation);
			CyrusPartition cyrusPartition = CyrusPartition.forServerAndDomain(mailbox.value.dataLocation, domainUid);
			String replicatedMailboxIdentifier = mailbox.value.type.nsPrefix + mailbox.value.name.replace(".", "^");
			IDbReplicatedMailboxes replicatedMailboxesService = context.provider()
					.instance(IDbReplicatedMailboxes.class, cyrusPartition.name, replicatedMailboxIdentifier);
			IDbMessageBodies messageBodiesService = context.provider().instance(IDbMessageBodies.class,
					cyrusPartition.name);

			ItemValue<Mailbox> mbox = context.provider().instance(IMailboxes.class, domainUid)
					.getComplete(entry.entryUid);

			replicatedMailboxesService.all().forEach(mailboxFolder -> {
				ItemRecords toRepair = repairFolder(logId, ds, domainUid, mbox, mailboxFolder, messageBodiesService);
				if (toRepair != null) {
					runOp.accept(toRepair);
				}
			});
			return true;
		}

		private ItemRecords repairFolder(String logId, DataSource ds, String domainUid, ItemValue<Mailbox> mbox,
				ItemValue<MailboxFolder> mailboxFolder, IDbMessageBodies messageBodiesService) {
			ContainerStore cs = new ContainerStore(context, ds, context.getSecurityContext());
			Container recordsContainer;
			try {
				recordsContainer = cs.get(IMailReplicaUids.mboxRecords(mailboxFolder.uid));
				if (recordsContainer == null) {
					logger.error("[{}] unable to find recordsContainer", logId);
					return null;
				}
			} catch (SQLException e) {
				logger.error("[{}] unable to find recordsContainer: {}", logId, e.getMessage());
				return null;
			}
			ItemRecords toRepair = new ItemRecords(domainUid, mbox, mailboxFolder.value.fullName, recordsContainer,
					messageBodiesService);
			try (Connection con = ds.getConnection();
					PreparedStatement stmt = con.prepareStatement(FIND_MISSING_BODIES_QUERY)) {
				stmt.setLong(1, recordsContainer.id);
				stmt.execute();
				try (ResultSet rs = stmt.getResultSet()) {
					while (rs.next()) {
						toRepair.add(rs.getInt(1), rs.getString(2));
					}
				}
			} catch (SQLException e) {
				logger.error("[{}] unable to acquire SQL connection: {}", logId, e.getMessage());
			}
			return toRepair;
		}

		private void repairRecordContainerItems(String logId, ItemRecords itemRecords) {
			logger.error("[{}] repairing folder {}: {} entries", logId, itemRecords.recordsContainer.name,
					itemRecords.size());
			if (itemRecords.shared()) {
				runOnSharedMailbox(logId, itemRecords);
			} else {
				runOnUserMailbox(logId, itemRecords);
			}
		}

		private void runOnUserMailbox(String logId, ItemRecords itemRecords) {
			String latd = itemRecords.mailbox.value.name + "@" + itemRecords.domainUid;
			LoginResponse resp = context.provider().instance(IAuthentication.class).su(latd);
			if (resp.authKey == null) {
				logger.error("[{}] sudo failed", logId);
				return;
			}
			ItemValue<Server> backend = itemRecords.getServer();
			try (StoreClient sc = new StoreClient(backend.value.address(), 1143, latd, resp.authKey)) {
				fixRecords(logId, sc, itemRecords);
			} catch (Exception e) {
				logger.error("[{}] unknown error: {}", logId, e.getMessage());
			}
		}

		private void fixRecords(String logId, StoreClient sc, ItemRecords itemRecords) {
			boolean loginOk = sc.login();
			if (!loginOk) {
				logger.error("[{}] failed to login", logId);
				return;
			}
			try {
				if (!sc.select(itemRecords.fullFolderName)) {
					logger.error("[{}] failed to select {}", logId, itemRecords.fullFolderName);
					return;
				}
			} catch (IMAPException e) {
				logger.error("[{}] failed to select {}: {}", logId, itemRecords.fullFolderName, e.getMessage());
				return;
			}

			// First, let's make sure the itemRecords are marked as deleted
			logger.info("[{}] repair of {} bodies", logId, itemRecords.size());
			itemRecords.records.forEach(ir -> {
				logger.info("[{}] trying to repair body {} in {}", logId, ir.bodyGuid, itemRecords.fullFolderName);
				IMAPByteSource bsrc = sc.uidFetchMessage(ir.imapUid);
				logger.info("[{}] got inputstream for {}", logId, ir.imapUid);
				try (InputStream src = bsrc.source().openStream()) {
					itemRecords.uploadBody(ir.bodyGuid, src);
				} catch (IOException ie) {
					logger.error("[{}] failed to get message {}/{}: {}", logId, itemRecords.fullFolderName, ir.imapUid,
							ie.getMessage());
				}
			});
		}

		private void runOnSharedMailbox(String logId, ItemRecords itemRecords) {
			ItemValue<Server> backend = itemRecords.getServer();

			try (StoreClient sc = new StoreClient(backend.value.address(), 1143, "admin0", Token.admin0())) {
				// FolderName on the shared mailbox is like "[sharedname]/[fullName]"
				itemRecords.fullFolderName += "@" + itemRecords.domainUid;
				logger.info("[{}] selected folder will be '{}'", logId, itemRecords.fullFolderName);
				fixRecords(logId, sc, itemRecords);
			} catch (Exception e) {
				logger.error("[{}] failed to repair mailshare {}: {}", logId, itemRecords.mailbox.uid, e.getMessage());
			}
		}
	}

	@Override
	public Set<MaintenanceOperation> availableOperations(Kind kind) {
		if (kind == Kind.USER || kind == Kind.MAILSHARE || kind == Kind.GROUP) {
			return ImmutableSet.of(MAINTENANCE_OPERATION);
		} else {
			return Collections.emptySet();
		}
	}

	@Override
	public Set<InternalMaintenanceOperation> ops(Kind kind) {
		if (kind == Kind.USER || kind == Kind.MAILSHARE || kind == Kind.GROUP) {
			return ImmutableSet.of(new BodiesMissingMaintenance(context));
		} else {
			return Collections.emptySet();
		}
	}
}
