/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.backend.mail.replica.service.internal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import net.bluemind.backend.mail.api.IMailboxFoldersByContainer;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.api.MailboxFolderSearchQuery;
import net.bluemind.backend.mail.api.utils.MailIndexQuery;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.mailbox.api.Mailbox.Type;

public class SearchQueryAdapter {

	private SearchQueryAdapter() {

	}

	public static MailIndexQuery adapt(BmContext context, String domainUid, String dirEntryUid,
			MailboxFolderSearchQuery query) {
		if (!isRecursive(query)) {
			return MailIndexQuery.simpleQuery(query);
		} else {
			ServerSideServiceProvider provider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
			DirEntry entry = provider.instance(IDirectory.class, domainUid).findByEntryUid(dirEntryUid);
			String subtree = null;
			if (entry.kind == Kind.MAILSHARE) {
				subtree = IMailReplicaUids.subtreeUid(domainUid, Type.mailshare, dirEntryUid);
			} else if (entry.kind == Kind.GROUP) {
				subtree = IMailReplicaUids.subtreeUid(domainUid, Type.group, dirEntryUid);
			} else {
				subtree = IMailReplicaUids.subtreeUid(domainUid, Type.user, dirEntryUid);
			}

			Set<String> folders = new HashSet<>();
			IServiceProvider contextProvider = ServerSideServiceProvider.getProvider(context);
			IMailboxFoldersByContainer service = contextProvider.instance(IMailboxFoldersByContainer.class, subtree);
			if (query.query.scope.folderScope != null && query.query.scope.folderScope.folderUid != null) {
				FolderTree fullTree = FolderTree.ofAll(service.all());
				folders.add(query.query.scope.folderScope.folderUid);
				ItemValue<MailboxFolder> complete = service.getComplete(query.query.scope.folderScope.folderUid);
				List<ItemValue<MailboxFolder>> children = fullTree.children(complete);
				folders.addAll(children.stream().map(f -> f.uid).collect(Collectors.toSet()));
			} else {
				folders.addAll(service.all().stream().map(f -> f.uid).collect(Collectors.toSet()));
			}
			return MailIndexQuery.folderQuery(query, new ArrayList<>(folders));
		}

	}

	private static boolean isRecursive(MailboxFolderSearchQuery query) {
		return query.query.scope != null && query.query.scope.isDeepTraversal;
	}

	static class FolderTree {
		private Map<String, Node> nodes = new HashMap<>();

		private static class Node {
			public final String uid;
			public final List<ItemValue<MailboxFolder>> children;

			public Node(ItemValue<MailboxFolder> f, List<ItemValue<MailboxFolder>> children) {
				this.uid = f.uid;
				this.children = children;
			}

		}

		static FolderTree ofAll(List<ItemValue<MailboxFolder>> folders) {
			FolderTree ft = new FolderTree();
			folders.stream().map(folder -> new Node(folder,
					folders.stream().filter(f -> folder.uid.equals(f.value.parentUid)).collect(Collectors.toList())))
					.forEach(n -> ft.nodes.put(n.uid, n));
			return ft;
		}

		List<ItemValue<MailboxFolder>> children(ItemValue<MailboxFolder> itemValue) {
			Node n = nodes.get(itemValue.uid);
			List<ItemValue<MailboxFolder>> ret = new ArrayList<>();
			children(ret, n);
			return ret;
		}

		private List<ItemValue<MailboxFolder>> children(List<ItemValue<MailboxFolder>> build, Node n) {
			n.children.forEach(child -> {
				build.add(child);
				children(build, nodes.get(child.uid));
			});
			return build;
		}
	}

}
