/**
 * BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

import axios from 'axios';

export default class {
    /**
     * Client implementation for net.bluemind.system.ldap.importation.api.ILdapImport API.
     * @param {string} sid Authentication token
     * @param {string} base RPC base path
     * @constructor
     */
    constructor(sid, base = '') {
        this.sid = sid;
        this.base = base + "/api/ldapimport";
    }
     

    /**
     * @param {string} hostname 
     * @param {string} protocol 
     * @param {string} allCertificate 
     * @param {string} basedn 
     * @param {string} logindn 
     * @param {string} password 
     * @param {string} userfilter 
     * @param {string} groupfilter 
     * @param {Object} canceller 
     * @return {Object}
     */
    testParameters(hostname, protocol, allCertificate, basedn, logindn, password, userfilter, groupfilter, canceller) {
        let $url = this.base + "/_testparameters";

        const $params = {};
        if (hostname != null) {
            Object.assign($params, {hostname});
        }
        if (protocol != null) {
            Object.assign($params, {protocol});
        }
        if (allCertificate != null) {
            Object.assign($params, {allCertificate});
        }
        if (basedn != null) {
            Object.assign($params, {basedn});
        }
        if (logindn != null) {
            Object.assign($params, {logindn});
        }
        if (password != null) {
            Object.assign($params, {password});
        }
        if (userfilter != null) {
            Object.assign($params, {userfilter});
        }
        if (groupfilter != null) {
            Object.assign($params, {groupfilter});
        }

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} uid 
     * @param {Object} canceller 
     * @return {Object}
     */
    fullSync(uid, canceller) {
        let $url = this.base + "/{uid}/_fullsync";
        $url = $url.replace('{uid}', uid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    success(response) {
        if (response.data && response.data.errorType) {
            throw {response};
        }
        return response.data;
    }
    
    failure(error) {
        if (axios.isCancel(error)) {
            error = { message: "CANCELLED_BY_CLIENT" };
        }

        let fail;
        if (error.response && error.response.data && error.response.data.errorType) {
            fail = new Error(error.response.data.message);
            fail.name = error.response.data.errorType;
            fail.data = error.response.data;
        } else if (error.response && error.response.message) {
            fail = new Error(error.response.message);
            fail.name = error.response.status;
            fail.data = error.response;
        } else {
            fail = new Error(error.message);
            fail.data = error;
        }
        throw fail;
    }
}
