/**
 * BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

import axios from 'axios';

export default class {
    /**
     * Client implementation for net.bluemind.system.api.hot.upgrade.IInternalHotUpgrade API.
     * @param {string} sid Authentication token
     * @param {string} base RPC base path
     * @constructor
     */
    constructor(sid, base = '') {
        this.sid = sid;
        this.base = base + "/api/hot_upgrade";
    }
     

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    running(canceller) {
        let $url = this.base + "/running";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    progress(canceller) {
        let $url = this.base + "/progress";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} filter 
     * @param {Object} canceller 
     * @return {Object}
     */
    list(filter, canceller) {
        let $url = this.base + "/list";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (filter != null) {
            $data = filter;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number} maxDuration 
     * @param {Object} mode 
     * @param {Object} canceller 
     * @return {Object}
     */
    startLimited(maxDuration, mode, canceller) {
        let $url = this.base + "/limitedStart";

        const $params = {};
        if (maxDuration != null) {
            Object.assign($params, {maxDuration});
        }

        let $data, $responseType, $acceptHeader;

        if (mode != null) {
            $data = mode;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} filter 
     * @param {Object} canceller 
     * @return {Object}
     */
    count(filter, canceller) {
        let $url = this.base + "/count";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (filter != null) {
            $data = filter;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {boolean} onlyReady 
     * @param {Object} mode 
     * @param {Object} canceller 
     * @return {Object}
     */
    start(onlyReady, mode, canceller) {
        let $url = this.base + "/start";

        const $params = {};
        if (onlyReady != null) {
            Object.assign($params, {onlyReady});
        }

        let $data, $responseType, $acceptHeader;

        if (mode != null) {
            $data = mode;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    success(response) {
        if (response.data && response.data.errorType) {
            throw {response};
        }
        return response.data;
    }
    
    failure(error) {
        if (axios.isCancel(error)) {
            error = { message: "CANCELLED_BY_CLIENT" };
        }

        let fail;
        if (error.response && error.response.data && error.response.data.errorType) {
            fail = new Error(error.response.data.message);
            fail.name = error.response.data.errorType;
            fail.data = error.response.data;
        } else if (error.response && error.response.message) {
            fail = new Error(error.response.message);
            fail.name = error.response.status;
            fail.data = error.response;
        } else {
            fail = new Error(error.message);
            fail.data = error;
        }
        throw fail;
    }
}
