/**
 * BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

import axios from 'axios';

export default class {
    /**
     * Client implementation for net.bluemind.system.api.IInstallation API.
     * @param {string} sid Authentication token
     * @param {string} base RPC base path
     * @constructor
     */
    constructor(sid, base = '') {
        this.sid = sid;
        this.base = base + "/api/system/installation";
    }
     

    /**
     * @param {number[]} logo 
     * @param {Object} canceller 
     * @return {Object}
     */
    setLogo(logo, canceller) {
        let $url = this.base + "/logo";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (logo != null) {
            $data = logo;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    getLogo(canceller) {
        let $url = this.base + "/logo";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    deleteLogo(canceller) {
        let $url = this.base + "/logo";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'DELETE',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    getSystemState(canceller) {
        let $url = this.base + "/state";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} ip 
     * @param {Object} canceller 
     * @return {Object}
     */
    ping(ip, canceller) {
        let $url = this.base + "/{ip}/ping";
        $url = $url.replace('{ip}', ip);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    removeSubscription(canceller) {
        let $url = this.base + "/subscription";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'DELETE',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    runningMode(canceller) {
        let $url = this.base + "/state/_maintenance";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'DELETE',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string[]} emails 
     * @param {Object} canceller 
     * @return {Object}
     */
    setSubscriptionContacts(emails, canceller) {
        let $url = this.base + "/_subscriptionContacts";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (emails != null) {
            $data = emails;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    getSubscriptionContacts(canceller) {
        let $url = this.base + "/_subscriptionContacts";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    markSchemaAsUpgraded(canceller) {
        let $url = this.base + "/version";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    postinst(canceller) {
        let $url = this.base + "/_postinst";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    upgradeStatus(canceller) {
        let $url = this.base + "/_upgrade";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    getSubscriptionKind(canceller) {
        let $url = this.base + "/subscriptionKind";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    getInfos(canceller) {
        let $url = this.base + "/_infos";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} index 
     * @param {Object} canceller 
     * @return {Object}
     */
    resetIndex(index, canceller) {
        let $url = this.base + "/{index}/_resetIndex";
        $url = $url.replace('{index}', index);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    resetIndexes(canceller) {
        let $url = this.base + "/_resetIndexes";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    sendHostReport(canceller) {
        let $url = this.base + "/_hostReport";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    isValidProductionSubscription(canceller) {
        let $url = this.base + "/subscriptionProductiveValid";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    getHostReport(canceller) {
        let $url = this.base + "/_hostReport";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    getVersion(canceller) {
        let $url = this.base + "/version";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    upgrade(canceller) {
        let $url = this.base + "/_upgrade";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    demoteLeader(canceller) {
        let $url = this.base + "/state/_demote";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} licence 
     * @param {Object} canceller 
     * @return {Object}
     */
    updateSubscription(licence, canceller) {
        let $url = this.base + "/subscription";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (licence != null) {
            $data = licence;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} version 
     * @param {Object} canceller 
     * @return {Object}
     */
    updateSubscriptionVersion(version, canceller) {
        let $url = this.base + "/subscription/_version";

        const $params = {};
        if (version != null) {
            Object.assign($params, {version});
        }

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {stream} archive 
     * @param {Object} canceller 
     * @param {Object} onUploadProgress 
     * @return {Object}
     */
    updateSubscriptionWithArchive(archive, canceller, onUploadProgress) {
        let $url = this.base + "/subscription/_archive";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (archive != null) {
            $data = archive;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            }),
            onUploadProgress
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} sourceParams 
     * @param {Object} canceller 
     * @return {Object}
     */
    clone(sourceParams, canceller) {
        let $url = this.base + "/_clone";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (sourceParams != null) {
            $data = sourceParams;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    getSubscriptionInformations(canceller) {
        let $url = this.base + "/subscription";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    initialize(canceller) {
        let $url = this.base + "/_initialize";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    maintenanceMode(canceller) {
        let $url = this.base + "/state/_maintenance";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    promoteLeader(canceller) {
        let $url = this.base + "/state/_promote";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    success(response) {
        if (response.data && response.data.errorType) {
            throw {response};
        }
        return response.data;
    }
    
    failure(error) {
        if (axios.isCancel(error)) {
            error = { message: "CANCELLED_BY_CLIENT" };
        }

        let fail;
        if (error.response && error.response.data && error.response.data.errorType) {
            fail = new Error(error.response.data.message);
            fail.name = error.response.data.errorType;
            fail.data = error.response.data;
        } else if (error.response && error.response.message) {
            fail = new Error(error.response.message);
            fail.name = error.response.status;
            fail.data = error.response;
        } else {
            fail = new Error(error.message);
            fail.data = error;
        }
        throw fail;
    }
}
