/**
 * BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

import axios from 'axios';

export default class {
    /**
     * Client implementation for net.bluemind.device.api.IDevice API.
     * @param {string} sid Authentication token
     * @param {string} userUid
     * @param {string} base RPC base path
     * @constructor
     */
    constructor(sid, userUid , base = '') {
        this.sid = sid;
        this.base = base + "/api/devices/{userUid}";
        this.userUid = userUid;
        this.base = this.base.replace('{userUid}',userUid);
    }
     

    /**
     * @param {string} uid 
     * @param {Object} canceller 
     * @return {Object}
     */
    setPartnership(uid, canceller) {
        let $url = this.base + "/{uid}/_partnership";
        $url = $url.replace('{uid}', uid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    deleteAll(canceller) {
        let $url = this.base + "/_deleteAll";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'DELETE',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} uid 
     * @param {Object} device 
     * @param {Object} canceller 
     * @return {Object}
     */
    update(uid, device, canceller) {
        let $url = this.base + "/{uid}";
        $url = $url.replace('{uid}', uid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (device != null) {
            $data = device;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    list(canceller) {
        let $url = this.base + "/_list";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} uid 
     * @param {Object} canceller 
     * @return {Object}
     */
    unwipe(uid, canceller) {
        let $url = this.base + "/_unwipe/{uid}";
        $url = $url.replace('{uid}', uid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} uid 
     * @param {Object} canceller 
     * @return {Object}
     */
    remove(uid, canceller) {
        let $url = this.base + "/{uid}";
        $url = $url.replace('{uid}', uid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'DELETE',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} uid 
     * @param {Object} canceller 
     * @return {Object}
     */
    unsetPartnership(uid, canceller) {
        let $url = this.base + "/{uid}/_partnership";
        $url = $url.replace('{uid}', uid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'DELETE',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} uid 
     * @param {Object} canceller 
     * @return {Object}
     */
    getComplete(uid, canceller) {
        let $url = this.base + "/{uid}/complete";
        $url = $url.replace('{uid}', uid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} uid 
     * @param {Object} canceller 
     * @return {Object}
     */
    wipe(uid, canceller) {
        let $url = this.base + "/_wipe/{uid}";
        $url = $url.replace('{uid}', uid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} identifier 
     * @param {Object} canceller 
     * @return {Object}
     */
    byIdentifier(identifier, canceller) {
        let $url = this.base + "/{identifier}/byIdentifier";
        $url = $url.replace('{identifier}', identifier);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} uid 
     * @param {Object} device 
     * @param {Object} canceller 
     * @return {Object}
     */
    create(uid, device, canceller) {
        let $url = this.base + "/{uid}";
        $url = $url.replace('{uid}', uid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (device != null) {
            $data = device;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} uid 
     * @param {Object} canceller 
     * @return {Object}
     */
    updateLastSync(uid, canceller) {
        let $url = this.base + "/{uid}/_lastSync";
        $url = $url.replace('{uid}', uid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    success(response) {
        if (response.data && response.data.errorType) {
            throw {response};
        }
        return response.data;
    }
    
    failure(error) {
        if (axios.isCancel(error)) {
            error = { message: "CANCELLED_BY_CLIENT" };
        }

        let fail;
        if (error.response && error.response.data && error.response.data.errorType) {
            fail = new Error(error.response.data.message);
            fail.name = error.response.data.errorType;
            fail.data = error.response.data;
        } else if (error.response && error.response.message) {
            fail = new Error(error.response.message);
            fail.name = error.response.status;
            fail.data = error.response;
        } else {
            fail = new Error(error.message);
            fail.data = error;
        }
        throw fail;
    }
}
