/**
 * BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

import axios from 'axios';

export default class {
    /**
     * Client implementation for net.bluemind.backend.mail.replica.api.IReplicatedMailboxesRootMgmt API.
     * @param {string} sid Authentication token
     * @param {string} partition
     * @param {string} base RPC base path
     * @constructor
     */
    constructor(sid, partition , base = '') {
        this.sid = sid;
        this.base = base + "/api/replicated_mailboxes_roots/{partition}";
        this.partition = partition;
        this.base = this.base.replace('{partition}',partition);
    }
     

    /**
     * @param {string} mailboxUid 
     * @param {boolean} setFlag 
     * @param {Object} canceller 
     * @return {Object}
     */
    resync(mailboxUid, setFlag, canceller) {
        let $url = this.base + "/_resync/{mailboxUid}";
        $url = $url.replace('{mailboxUid}', mailboxUid);

        const $params = {};
        if (setFlag != null) {
            Object.assign($params, {setFlag});
        }

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} rename 
     * @param {Object} canceller 
     * @return {Object}
     */
    update(rename, canceller) {
        let $url = this.base + "/_update";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (rename != null) {
            $data = rename;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} namespace 
     * @param {string} mailboxName 
     * @param {Object} canceller 
     * @return {Object}
     */
    remove(namespace, mailboxName, canceller) {
        let $url = this.base + "/_delete/{namespace}/{mailboxName}";
        $url = $url.replace('{namespace}', namespace);
        $url = $url.replace('{mailboxName}', mailboxName);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'DELETE',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} root 
     * @param {Object} canceller 
     * @return {Object}
     */
    create(root, canceller) {
        let $url = this.base + "/_create";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (root != null) {
            $data = root;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    success(response) {
        if (response.data && response.data.errorType) {
            throw {response};
        }
        return response.data;
    }
    
    failure(error) {
        if (axios.isCancel(error)) {
            error = { message: "CANCELLED_BY_CLIENT" };
        }

        let fail;
        if (error.response && error.response.data && error.response.data.errorType) {
            fail = new Error(error.response.data.message);
            fail.name = error.response.data.errorType;
            fail.data = error.response.data;
        } else if (error.response && error.response.message) {
            fail = new Error(error.response.message);
            fail.name = error.response.status;
            fail.data = error.response;
        } else {
            fail = new Error(error.message);
            fail.data = error;
        }
        throw fail;
    }
}
