/**
 * BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

import axios from 'axios';

export default class {
    /**
     * Client implementation for net.bluemind.backend.mail.api.IMailboxItems API.
     * @param {string} sid Authentication token
     * @param {string} replicatedMailboxUid
     * @param {string} base RPC base path
     * @constructor
     */
    constructor(sid, replicatedMailboxUid , base = '') {
        this.sid = sid;
        this.base = base + "/api/mail_items/{replicatedMailboxUid}";
        this.replicatedMailboxUid = replicatedMailboxUid;
        this.base = this.base.replace('{replicatedMailboxUid}',replicatedMailboxUid);
    }
     

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    getVersion(canceller) {
        let $url = this.base + "/_version";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number} since 
     * @param {Object} arg1 
     * @param {Object} canceller 
     * @return {Object}
     */
    filteredChangesetById(since, arg1, canceller) {
        let $url = this.base + "/_filteredChangesetById";

        const $params = {};
        if (since != null) {
            Object.assign($params, {since});
        }

        let $data, $responseType, $acceptHeader;

        if (arg1 != null) {
            $data = arg1;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} uid 
     * @param {number} arg1 
     * @param {Object} canceller 
     * @return {Object}
     */
    itemChangelog(uid, arg1, canceller) {
        let $url = this.base + "/{uid}/_itemchangelog";
        $url = $url.replace('{uid}', uid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (arg1 != null) {
            $data = arg1;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number} since 
     * @param {Object} canceller 
     * @return {Object}
     */
    changeset(since, canceller) {
        let $url = this.base + "/_changeset";

        const $params = {};
        if (since != null) {
            Object.assign($params, {since});
        }

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number} since 
     * @param {Object} canceller 
     * @return {Object}
     */
    changesetById(since, canceller) {
        let $url = this.base + "/_changesetById";

        const $params = {};
        if (since != null) {
            Object.assign($params, {since});
        }

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number} arg0 
     * @param {Object} canceller 
     * @return {Object}
     */
    containerChangelog(arg0, canceller) {
        let $url = this.base + "/_changelog";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (arg0 != null) {
            $data = arg0;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} filter 
     * @param {number} knownContainerVersion 
     * @param {number} limit 
     * @param {number} offset 
     * @param {Object} canceller 
     * @return {Object}
     */
    allIds(filter, knownContainerVersion, limit, offset, canceller) {
        let $url = this.base + "/_itemIds";

        const $params = {};
        if (filter != null) {
            Object.assign($params, {filter});
        }
        if (knownContainerVersion != null) {
            Object.assign($params, {knownContainerVersion});
        }
        if (limit != null) {
            Object.assign($params, {limit});
        }
        if (offset != null) {
            Object.assign($params, {offset});
        }

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number[]} arg0 
     * @param {Object} canceller 
     * @return {Object}
     */
    multipleGetById(arg0, canceller) {
        let $url = this.base + "/_mgetById";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (arg0 != null) {
            $data = arg0;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number} id 
     * @param {Object} canceller 
     * @return {Object}
     */
    getCompleteById(id, canceller) {
        let $url = this.base + "/{id}/completeById";
        $url = $url.replace('{id}', id);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number[]} arg0 
     * @param {Object} canceller 
     * @return {Object}
     */
    multipleDeleteById(arg0, canceller) {
        let $url = this.base + "/_multipleDelete";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (arg0 != null) {
            $data = arg0;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'DELETE',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number} id 
     * @param {Object} arg1 
     * @param {Object} canceller 
     * @return {Object}
     */
    updateById(id, arg1, canceller) {
        let $url = this.base + "/id/{id}";
        $url = $url.replace('{id}', id);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (arg1 != null) {
            $data = arg1;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number} id 
     * @param {Object} canceller 
     * @return {Object}
     */
    deleteById(id, canceller) {
        let $url = this.base + "/id/{id}";
        $url = $url.replace('{id}', id);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'DELETE',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number} id 
     * @param {Object} arg1 
     * @param {Object} canceller 
     * @return {Object}
     */
    createById(id, arg1, canceller) {
        let $url = this.base + "/id/{id}";
        $url = $url.replace('{id}', id);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (arg1 != null) {
            $data = arg1;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} arg0 
     * @param {Object} canceller 
     * @return {Object}
     */
    count(arg0, canceller) {
        let $url = this.base + "/_count";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (arg0 != null) {
            $data = arg0;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} arg0 
     * @param {Object} canceller 
     * @return {Object}
     */
    sortedIds(arg0, canceller) {
        let $url = this.base + "/_sorted";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (arg0 != null) {
            $data = arg0;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number} id 
     * @param {Object} canceller 
     * @return {Object}
     */
    getForUpdate(id, canceller) {
        let $url = this.base + "/{id}/getForUpdate";
        $url = $url.replace('{id}', id);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} flagUpdate 
     * @param {Object} canceller 
     * @return {Object}
     */
    deleteFlag(flagUpdate, canceller) {
        let $url = this.base + "/_deleteFlag";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (flagUpdate != null) {
            $data = flagUpdate;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    unreadItems(canceller) {
        let $url = this.base + "/_unread";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {stream} part 
     * @param {Object} canceller 
     * @param {Object} onUploadProgress 
     * @return {Object}
     */
    uploadPart(part, canceller, onUploadProgress) {
        let $url = this.base + "/_part";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (part != null) {
            $data = part;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            }),
            onUploadProgress
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} value 
     * @param {Object} canceller 
     * @return {Object}
     */
    create(value, canceller) {
        let $url = this.base;

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (value != null) {
            $data = value;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {date} deliveredOrUpdatedAfter 
     * @param {Object} canceller 
     * @return {Object}
     */
    recentItems(deliveredOrUpdatedAfter, canceller) {
        let $url = this.base + "/_recent";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (deliveredOrUpdatedAfter != null) {
            $data = deliveredOrUpdatedAfter;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number} itemId 
     * @param {Object} canceller 
     * @return {Object}
     */
    unexpunge(itemId, canceller) {
        let $url = this.base + "/_unexpunge/{itemId}";
        $url = $url.replace('{itemId}', itemId);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    expunge(canceller) {
        let $url = this.base + "/_expunge";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number} imapUid 
     * @param {Object} canceller 
     * @return {Object}
     */
    fetchComplete(imapUid, canceller) {
        let $url = this.base + "/eml/{imapUid}";
        $url = $url.replace('{imapUid}', imapUid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "blob";
        $acceptHeader = "*/*";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} flagUpdate 
     * @param {Object} canceller 
     * @return {Object}
     */
    addFlag(flagUpdate, canceller) {
        let $url = this.base + "/_addFlag";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (flagUpdate != null) {
            $data = flagUpdate;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} partId 
     * @param {Object} canceller 
     * @return {Object}
     */
    removePart(partId, canceller) {
        let $url = this.base + "/{partId}/_part";
        $url = $url.replace('{partId}', partId);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'DELETE',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number[]} ids 
     * @param {Object} canceller 
     * @return {Object}
     */
    multipleById(ids, canceller) {
        let $url = this.base + "/_multipleById";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (ids != null) {
            $data = ids;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number} imapUid 
     * @param {string} address 
     * @param {string} encoding 
     * @param {string} mime 
     * @param {string} charset 
     * @param {string} filename 
     * @param {Object} canceller 
     * @return {Object}
     */
    fetch(imapUid, address, encoding, mime, charset, filename, canceller) {
        let $url = this.base + "/part/{imapUid}/{address}";
        $url = $url.replace('{imapUid}', imapUid);
        $url = $url.replace('{address}', address);

        const $params = {};
        if (encoding != null) {
            Object.assign($params, {encoding});
        }
        if (mime != null) {
            Object.assign($params, {mime});
        }
        if (charset != null) {
            Object.assign($params, {charset});
        }
        if (filename != null) {
            Object.assign($params, {filename});
        }

        let $data, $responseType, $acceptHeader;

        $responseType = "blob";
        $acceptHeader = "*/*";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    success(response) {
        if (response.data && response.data.errorType) {
            throw {response};
        }
        return response.data;
    }
    
    failure(error) {
        if (axios.isCancel(error)) {
            error = { message: "CANCELLED_BY_CLIENT" };
        }

        let fail;
        if (error.response && error.response.data && error.response.data.errorType) {
            fail = new Error(error.response.data.message);
            fail.name = error.response.data.errorType;
            fail.data = error.response.data;
        } else if (error.response && error.response.message) {
            fail = new Error(error.response.message);
            fail.name = error.response.status;
            fail.data = error.response;
        } else {
            fail = new Error(error.message);
            fail.data = error;
        }
        throw fail;
    }
}
