/**
 * BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

import axios from 'axios';

export default class {
    /**
     * Client implementation for net.bluemind.backend.mail.api.IItemsTransfer API.
     * @param {string} sid Authentication token
     * @param {string} fromMailboxUid
     * @param {string} toMailboxUid
     * @param {string} base RPC base path
     * @constructor
     */
    constructor(sid, fromMailboxUid , toMailboxUid , base = '') {
        this.sid = sid;
        this.base = base + "/api/mail_items_transfer/{fromMailboxUid}/{toMailboxUid}";
        this.fromMailboxUid = fromMailboxUid;
        this.base = this.base.replace('{fromMailboxUid}',fromMailboxUid);
        this.toMailboxUid = toMailboxUid;
        this.base = this.base.replace('{toMailboxUid}',toMailboxUid);
    }
     

    /**
     * @param {number[]} itemIds 
     * @param {Object} canceller 
     * @return {Object}
     */
    move(itemIds, canceller) {
        let $url = this.base + "/move";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (itemIds != null) {
            $data = itemIds;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {number[]} itemIds 
     * @param {Object} canceller 
     * @return {Object}
     */
    copy(itemIds, canceller) {
        let $url = this.base + "/copy";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (itemIds != null) {
            $data = itemIds;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    success(response) {
        if (response.data && response.data.errorType) {
            throw {response};
        }
        return response.data;
    }
    
    failure(error) {
        if (axios.isCancel(error)) {
            error = { message: "CANCELLED_BY_CLIENT" };
        }

        let fail;
        if (error.response && error.response.data && error.response.data.errorType) {
            fail = new Error(error.response.data.message);
            fail.name = error.response.data.errorType;
            fail.data = error.response.data;
        } else if (error.response && error.response.message) {
            fail = new Error(error.response.message);
            fail.name = error.response.status;
            fail.data = error.response;
        } else {
            fail = new Error(error.message);
            fail.data = error;
        }
        throw fail;
    }
}
